////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////

package com.crankuptheamps.client;
import com.crankuptheamps.client.exception.*;
import java.io.*;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileChannel.MapMode;


/**
 * PublishStore is a memory-mapped file-backed store for storing outgoing messages.
 * As messages are stored, space is allocated from a pre-created flat buffer on disk.
 * As messages are discarded, space in that buffer is marked "free" for future store operations.
 * If messages are stored faster than they are published, the buffer is re-sized to create more capacity.
 * 
 * @author dnoor
 *
 */
public class PublishStore extends BlockPublishStore
{
    final static int DEFAULT_INITIAL_BLOCKS = 10 * 1000;
    static class MMapStoreBuffer extends MemoryStoreBuffer
    {
        String _path;
        RandomAccessFile _file;
        FileChannel _channel;
        Store _store;

        public MMapStoreBuffer(String path) throws StoreException
        {
            _path = path;
            try {
                _file = new RandomAccessFile(path, "rw");
            } catch (IOException e) {
                throw new StoreException(e);
            }
            _channel = _file.getChannel();
        }
        
        @Override
        public long getSize() throws IOException
        {
            if(_buffer == null)
            {
                if(_file.length() == 0)
                {
                    return 0;
                }
                _buffer = _channel.map(MapMode.READ_WRITE, 0, _file.length());
            }
            return _buffer.capacity();
        }
        
        @Override
        public void setSize(long newSize) throws IOException
        {
            if(_channel == null)
            {
                throw new IOException("The store is closed.");
            }
            if(_buffer != null) ((MappedByteBuffer)_buffer).force();
            if(_resizeHandler != null && !_resizeHandler.invoke(_store, newSize))
                return;
            _buffer = _channel.map(MapMode.READ_WRITE, 0, newSize);
        }
        
        public void close() throws IOException
        {
            _buffer = null;
            _channel.close();
            _file.close();
            
        }
        
        public void sync() throws IOException
        {
            MappedByteBuffer b = ((MappedByteBuffer)_buffer);
            b.force();
            
        }
        
        @Override
        public void setResizeHandler(PublishStoreResizeHandler handler, Store store)
        {
            _resizeHandler = handler;
            _store = store;
        }
        
        @Override
        protected void finalize() throws Throwable
        {
            close();
        }
    }

    /**
     * Creates a new PublishStore with the given path.  Immediately proceeds to recovery if the file exists. 
     * @param path The path (absolute or relative) of the publish store
     * @throws IOException
     */
    public PublishStore(String path) throws StoreException
    {
        this(path, DEFAULT_INITIAL_BLOCKS);
    }

    /**
     * Creates a new PublishStore with the given path.  Immediately proceeds to recovery if the file exists.
     * @param path The path (absolute or relative) of the publish store
     * @param initialCapacity The initial capacity (in 2k blocks) of the store.  This size is also used when
     *     resizing the store.
     * @throws IOException
     */
    public PublishStore(String path, int initialCapacity) throws StoreException
    {
        super(new MMapStoreBuffer(path), initialCapacity);
        recover();
        if (_usedList == null)
        {
            growFreeListIfEmpty();
        }
    }
    public void close() throws IOException
    {
        ((MMapStoreBuffer)_buffer).close();
    
    }
    public void sync() throws IOException
    {
        ((MMapStoreBuffer)_buffer).sync();
    
    }
    @Override
    protected void finalize() throws Throwable
    {
        close();
    }
}

