////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////
package com.crankuptheamps.client;

import java.io.Closeable;
import java.util.Iterator;
import java.util.concurrent.ConcurrentLinkedQueue;
import com.crankuptheamps.client.exception.*;

/**
 * MessageStream provides an iteration abstraction over the results of an AMPS command such
 * as a subscribe, a SOW query, or SOW delete. MessageStream is produced when calling Client.execute() and
 * continues iterating over the results until the connection is closed, or the iterator is explicitly closed, or when the SOW query is ended.
 * 
 * You can use a MessageStream as you would other Iterators, for example, using a for loop (Java 1.7):
 * <pre><code>
 * MessageStream stream = client.execute(new Command("sow").setTopic("/orders"));
 * for(Message message : stream)
 * {
 *   ...
 * }
 * </pre></code>
 */
public class MessageStream implements Iterator<Message>, Iterable<Message>, MessageHandler, ConnectionStateListener, Closeable
{
    CommandId _commandId;
    Client _client;
    ConcurrentLinkedQueue<Message> _q = new ConcurrentLinkedQueue<Message>();
    Message _current = null;
    final int STATE_Unset = 0x0;
    final int STATE_Reading = 0x10;
    final int STATE_Subscribed = 0x11;
    final int STATE_SOWOnly = 0x12;
    final int STATE_StatsOnly = 0x13;
    final int STATE_Disconnected = 0x01;
    final int STATE_Complete = 0x02;
    int _state = STATE_Unset;
    
    protected MessageStream(Client client_)
    {
        _client = client_;
        _client.addConnectionStateListener(this);
    }
    protected void setSubscription(CommandId commandId_)
    {
        _commandId = commandId_;
        _state = STATE_Subscribed;
    }
    protected void setSOWOnly()
    {
        _state = STATE_SOWOnly;
    }
    protected void setStatsOnly()
    {
        _state = STATE_StatsOnly;
    }
    protected void setRunning()
    {
        _state = STATE_Reading;
    }
    
    @Override
    public void connectionStateChanged(int newState_)
    {
       if(newState_ == Disconnected)
       {
           _state = STATE_Disconnected;
       }
    }
    
    @Override
    public boolean hasNext()
    {
        if(_current != null) return true;
        
        _current = _q.poll();
        while((_state & STATE_Reading) > 0 && _current==null)
        {
            Thread.yield();
            _current = _q.poll();
        }

        // NOTE: A DisconnectedException cannot be thrown from an iterator's hasNext.
        return _current != null;
    }

    @Override
    public Message next()
    {
        if(_current == null) hasNext();
        Message retVal = _current;
        _current = null;
        if(_state==STATE_SOWOnly && retVal != null && retVal.getCommand() == Message.Command.GroupEnd)
        {
            _state = STATE_Complete;
        } else if (_state==STATE_StatsOnly && retVal!=null && retVal.getAckType() == Message.AckType.Stats)
        {
            _state = STATE_Complete;
        }
        return retVal;
    }

    @Override
    public void invoke(Message message)
    {
        _q.add(message.copy());
    }

    public void remove()
    {
        throw new UnsupportedOperationException();
    }
    
    public Iterator<Message> iterator()
    {
        return this;
    }
    
    /**
     * @return true if the connection to AMPS is still active, or false if a disconnect is detected.
     */
    public boolean isConnected()
    { 
        return _state != STATE_Disconnected;
    }
    
    /**
     * Closes this MessageStream, unsubscribing from AMPS if applicable.
     */
    public void close()
    {
        try
        {
            if(_client != null)
            {
                _client.removeConnectionStateListener(this);
                if(_state == STATE_Subscribed && _commandId != null) _client.unsubscribe(_commandId);
            }
        } catch (Throwable t)
        {
        }
        _state = STATE_Complete;
    }
}
