/////   ///////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////

package com.crankuptheamps.client;
import java.io.IOException;

import com.crankuptheamps.client.exception.DisconnectedException;
import com.crankuptheamps.client.exception.StoreException;
import com.crankuptheamps.client.exception.TimedOutException;

/**
 * PublishStore that stores first in memory, and swaps excess out to disk.
 *
 */
public class HybridPublishStore implements Store
{
    protected PublishStore _fileStore;
    protected MemoryPublishStore _memoryStore;
    protected int _cap;
    protected String _path;
    protected int _lowWatermark = 0;
    protected long _lowestIndexInTheFile = 0;
    
    private static class SwappingOutReplayer implements Store.StoreReplayer
    {
        HybridPublishStore _store;
        int _entries, _errorCount;
        long _lastIndex;
        private SwappingOutReplayer(HybridPublishStore store_, int entries_)
        {
            _store = store_;
            _entries = entries_;
        }
        public int getErrors()
        {
            return _errorCount;
        }
        public long lastIndex()
        {
            return _lastIndex;
        }

        public void execute(long index, int operation, byte[] topic,
                long topicOffset, long topicLength, byte[] data,
                long dataOffset, long dataLength, byte[] corId, long corIdOff, long corIdLen)
        {
            if(_entries > 0 && _errorCount == 0)
            {
                try
                {
                    _store._fileStore.store(index, operation, topic, topicOffset,
                                            topicLength, data, dataOffset, dataLength, corId, corIdOff, corIdLen);
                    _lastIndex = index;
                } catch (Exception e)
                {
                    ++_errorCount;
                }
                --_entries;
            }
        }

        public void execute(long index, int operation, byte[] topic,
                long topicOffset, long topicLength, byte[] data,
                long dataOffset, long dataLength, byte[] corId, long corIdOff, long corIdLen, 
                int expiration)
        {
            if(_entries > 0 && _errorCount == 0)
            {
                try
                {
                    _store._fileStore.store(index, operation, topic, topicOffset,
                                            topicLength, data, dataOffset,
                                            dataLength, corId, corIdOff, corIdLen, expiration);
                    _lastIndex = index;
                } catch (Exception e)
                {
                    ++_errorCount;
                }
                --_entries;
            }
        }

        public void execute(long index, int operation, byte[] topic,
                long topicOffset, long topicLength, byte[] data,
                long dataOffset, long dataLength, int expiration, CommandId cmdId)
        {
            if(_entries > 0 && _errorCount == 0)
            {
                try
                {
                    _store._fileStore.store(index, operation, topic, topicOffset,
                                            topicLength, data, dataOffset,
                                            dataLength, expiration, cmdId);
                    _lastIndex = index;
                } catch (Exception e)
                {
                    ++_errorCount;
                }
                --_entries;
            }
        }
    }
    public HybridPublishStore(String path, int cap) throws IOException, StoreException
    {
        _fileStore = new PublishStore(path);
        _memoryStore = new MemoryPublishStore(cap);
        _cap = cap;
        _lowWatermark = (int)(_cap * 0.5);
        _path = path;
    }
    
    /**
     * Sets the low watermark; once we start swapping out to disk, we 
     * keep going until the number of entries in memory is lower than this.
     * @param lowWatermark_ The number of entries to serve as a low watermark.
     */
    public void setLowWatermark(int lowWatermark_)
    {
        if(lowWatermark_ < _cap)
        {
            _lowWatermark = lowWatermark_;
        }
    }

    /* Returns the lowest sequence number currently in the store.
     */
    public synchronized long getLowestUnpersisted()
    {
        if (_fileStore.unpersistedCount() == 0)
            return _memoryStore.getLowestUnpersisted();
        return _fileStore.getLowestUnpersisted();
    }

    /* Discards from both the disk store and memory store.
     * @see com.crankuptheamps.client.Store#discardUpTo(long)
     */
    public synchronized void discardUpTo(long index) throws StoreException
    {
        _fileStore.discardUpTo(index);
        _memoryStore.discardUpTo(index);

    }
    
    /* 
     * @see com.crankuptheamps.client.Store#replay(com.crankuptheamps.client.Store.StoreReplayer)
     */
    public void replay(StoreReplayer replayer) throws StoreException, DisconnectedException
    {
        // First replay everything on disk, then replay everything in memory.  Simple!
        _fileStore.replay(replayer);
        _memoryStore.replay(replayer);
    }

    /* 
     * @see com.crankuptheamps.client.Store#unpersistedCount()
     */
    public synchronized long unpersistedCount()
    {
        return _fileStore.unpersistedCount() + _memoryStore.unpersistedCount();
    }
    public void replaySingle(StoreReplayer replayer, long index)
            throws StoreException, DisconnectedException
    {
        if(index <= _lowestIndexInTheFile)
        {
            _fileStore.replaySingle(replayer, index);
        }
        else
        {
            _memoryStore.replaySingle(replayer, index);
        }
        
    }

    public void store(long index, int operation, byte[] topic,
            long topicOffset, long topicLength, byte[] data, long dataOffset,
            long dataLength, byte[] corId, long corIdOff, long corIdLen) throws StoreException
    {
        try
        {
            if(_memoryStore.unpersistedCount() >= _cap)
            {
                // ok, swap some out to disk now.
                SwappingOutReplayer replayer = new SwappingOutReplayer(this, (int)_memoryStore.unpersistedCount() - _lowWatermark);
                _memoryStore.replay(replayer);
                if(replayer.getErrors() == 0)
                {
                    _memoryStore.discardUpTo(replayer.lastIndex());
                    _lowestIndexInTheFile = replayer.lastIndex();
                }
            }
            _memoryStore.store(index, operation, topic, topicOffset, topicLength, 
                data, dataOffset, dataLength, corId, corIdOff, corIdLen);
        }
        catch(DisconnectedException e)
        {
            // This "cannot" happen, as our replayer does not throw DisconnectedException
            assert false;
        }
    }

    public void store(long index, int operation, byte[] topic,
            long topicOffset, long topicLength, byte[] data, long dataOffset,
            long dataLength, byte[] corId, long corIdOff, long corIdLen, int expiration) throws StoreException
    {
        try
        {
            if(_memoryStore.unpersistedCount() >= _cap)
            {
                // ok, swap some out to disk now.
                SwappingOutReplayer replayer = new SwappingOutReplayer(this, (int)_memoryStore.unpersistedCount() - _lowWatermark);
                _memoryStore.replay(replayer);
                if(replayer.getErrors() == 0)
                {
                    _memoryStore.discardUpTo(replayer.lastIndex());
                    _lowestIndexInTheFile = replayer.lastIndex();
                }
            }
            _memoryStore.store(index, operation, topic, topicOffset, topicLength, 
                data, dataOffset, dataLength, corId, corIdOff, corIdLen, expiration);
        }
        catch(DisconnectedException e)
        {
            // This "cannot" happen, as our replayer does not throw DisconnectedException
            assert false;
        }
    }

    public void store(long index, int operation, byte[] topic,
            long topicOffset, long topicLength, byte[] data, long dataOffset,
            long dataLength, int expiration, CommandId cmdId) throws StoreException
    {
        try
        {
            if(_memoryStore.unpersistedCount() >= _cap)
            {
                // ok, swap some out to disk now.
                SwappingOutReplayer replayer = new SwappingOutReplayer(this, (int)_memoryStore.unpersistedCount() - _lowWatermark);
                _memoryStore.replay(replayer);
                if(replayer.getErrors() == 0)
                {
                    _memoryStore.discardUpTo(replayer.lastIndex());
                    _lowestIndexInTheFile = replayer.lastIndex();
                }
            }
            _memoryStore.store(index, operation, topic, topicOffset, topicLength, 
                data, dataOffset, dataLength, expiration, cmdId);
        }
        catch(DisconnectedException e)
        {
            // This "cannot" happen, as our replayer does not throw DisconnectedException
            assert false;
        }
    }

    public synchronized void flush() throws TimedOutException
    {
        if (_memoryStore.unpersistedCount() == 0 && _fileStore.unpersistedCount() == 0)
            return;
        long current = _memoryStore._endOfUsedList.sequenceNumber;
        boolean keepWaiting = true;
        while (keepWaiting)
        {
            keepWaiting = _memoryStore._usedList == null ? false :
                          current >= _memoryStore._usedList.sequenceNumber;
            if (!keepWaiting)
               keepWaiting = _fileStore._usedList == null ? false :
                             current >= _fileStore._usedList.sequenceNumber;
            try
            {
                if (keepWaiting) this.wait();
            } catch (InterruptedException e)
            {
            }
        }
    }

    public synchronized void flush(long timeout) throws TimedOutException
    {
        if (_memoryStore.unpersistedCount() == 0 && _fileStore.unpersistedCount() == 0)
            return;
        long current = _memoryStore._endOfUsedList.sequenceNumber;
        boolean keepWaiting = true;
        long start = System.currentTimeMillis();
        long end = System.currentTimeMillis();
        while (keepWaiting)
        {
            keepWaiting = _memoryStore._usedList == null ? false :
                          current >= _memoryStore._usedList.sequenceNumber;
            if (!keepWaiting)
               keepWaiting = _fileStore._usedList == null ? false :
                             current >= _fileStore._usedList.sequenceNumber;
            try
            {
                long remaining = timeout - (end - start);
                if (keepWaiting) this.wait(remaining);
                end = System.currentTimeMillis();
                if ((end - start) >= timeout)
                    throw new TimedOutException("Timed out waiting to flush publish store.");
            } catch (InterruptedException e)
            {
            }
        }
    }

    public void setResizeHandler(PublishStoreResizeHandler handler)
    {
        _fileStore.setResizeHandler(handler);
        _memoryStore.setResizeHandler(handler);
    }
}
