////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////
package com.crankuptheamps.client;

import com.crankuptheamps.client.fields.CommandField;
/**
 * 
 * Command is an encapsulation of a single AMPS command sent by the client.
 * Using Command you can build  valid commands to be executed either
 * synchronously or asynchronously via the {@link Client} execute() and
 * executeAsync() methods.
 * Command is designed to be used as a "builder" enabling AMPS commands
 * to be built easily, for example:
 * 
 * <pre><code>
 * Client client = new Client(...);
 * for(Message m : client.execute(new Command("sow").setTopic("topic"))) { ... }
 * </pre></code>
 *
 * @since 4.0.0.0
 */
public class Command
{
    CommandId CommandId,QueryId,SubId;
    int Command, AckType;
    String Filter, OrderBy, Bookmark, Options, SOWKeys;
    byte[] Topic, Data;
    String TopicString, DataString;
    String CorrelationId;
    int _topicLength, _topicOffset, _dataLength, _dataOffset;
    long Timeout = 0;
    int TopN = -1 , BatchSize = 1;
    int Expiration = 0;
    boolean _expiration_isSet = false;
    long ClientSequenceNumber =0;
    Message _message=null;

    private void init()
    {
        CommandId = QueryId = SubId = null;
        Filter = OrderBy = Bookmark = Options = SOWKeys = CorrelationId = null;
        Topic = Data = null;
        TopicString = DataString = null;
        
        AckType = Expiration = 0;
        TopN = -1;
        BatchSize = 1;
        Timeout = ClientSequenceNumber = 0;
        _expiration_isSet = false;
        
        if(Command != Message.Command.Publish && Command != Message.Command.DeltaPublish && Command != Message.Command.StartTimer)
        {
            CommandId = com.crankuptheamps.client.CommandId.nextIdentifier();
            if(isSow())
            {
                QueryId = CommandId;
                if(Command == Message.Command.SOW)
                {
                    AckType |= Message.AckType.Completed;
                }
            }
            if(isSubscribe())
            {
                SubId = CommandId;
            }
            AckType |= Message.AckType.Processed;
        }  
    }
    
    /**
     * Returns the SOW keys on this command.
     * @return the sow keys
     * @since 4.0.0.0
     */
    public String getSOWKeys()
    {
        return SOWKeys;
    }

    /**
     * The SowKeys for a command are a comma-separated list
     * of the keys that AMPS assigns to SOW messages. The SOW key for a
     * message is available through the {@link Message#getSowKey} method
     * on a message.
     *
     * @param sowKeys the sow keys to set
     * @since 4.0.0.0
     */
    public Command setSOWKeys(String sowKeys)
    {
        SOWKeys = sowKeys;
        return this;
    }

    /**
     * Returns true if this command creates a subscription.
     * @return true if this command creates a subscription.
     * @since 4.0.0.0
     */
    public boolean isSubscribe()
    {
        return Command == Message.Command.DeltaSubscribe || Command == Message.Command.SOWAndDeltaSubscribe ||
               Command == Message.Command.SOWAndSubscribe || Command == Message.Command.Subscribe;
    }
    
    /**
     * Returns true if this command queries a SOW.
     * @return true if this command queries a SOW.
     * @since 4.0.0.0
     */
    public boolean isSow()
    {
        return Command == Message.Command.SOW || Command == Message.Command.SOWAndDeltaSubscribe ||
                Command == Message.Command.SOWAndSubscribe;
    }
    /**
     * Returns true if this command can be associated with a client sequence number.
     * 
     * @return true if this command can be associated with a client sequence number.
     * @since 4.0.0.0
     */
    public boolean needsSequenceNumber()
    {
        return Command == Message.Command.Publish || Command == Message.Command.DeltaPublish || Command == Message.Command.SOWDelete;
    }
    private int translateCommand(String command_)
    throws IllegalArgumentException
    {
        CommandField cf = new CommandField();
        cf.set(command_.getBytes());
        if(!CommandField.encodeCommand(cf.getValue()).equals(command_))
        {
            throw new IllegalArgumentException("unknown command type " + command_);
        }
        return cf.getValue();
    }
    /**
     * Create a new Command with no command type or arguments set.
     * @since 4.0.0.0
     */
    public Command()
    {
    }
    /**
     * Create a Command with the Command field set.
     * @param command_ A {@link Message.Command} value indicating the AMPS command.
     * @since 4.0.0.0
     */
    public Command(int command_)
    {
        Command = command_;
        init();
    }
    /**
     * Create a Command with the Command field set.
     * @param command_ A string indicating the AMPS command.
     * @since 4.0.0.0
     */
    public Command(String command_)
    {
        Command = translateCommand(command_);
        init();
    }
    /**
     * Resets this command with a new Command type and re-initializes all other fields.
     * @param command_ A {@link Message.Command} value indicating the AMPS command.
     * @since 4.0.0.0
     */
    public Command reset(int command_)
    {
        Command = command_;
        init();
        return this;
    }
    /**
     * Resets this command with a new Command type and re-initializes all other fields.
     * @param command_ A string value indicating the AMPS command.
     * @since 4.0.0.0
     */
    public Command reset(String command_)
    {
        Command = translateCommand(command_);
        init();
        return this;
    }
    /**
     * @return the commandId, which is automatically generated for some command types.
     * @since 4.0.0.0
     */
    public CommandId getCommandId()
    {
        return CommandId;
    }
    /**
     * Set the commandId for this command.
     * @param commandId the commandId to set
     * @since 4.0.0.0
     */
    public Command setCommandId(CommandId commandId)
    {
        CommandId = commandId;
        return this;
    }
    /**
     * Get the queryId for this command.
     * @return the queryId
     * @since 4.0.0.0
     */
    public CommandId getQueryId()
    {
        return QueryId;
    }
    /**
     * Set the queryId for this command.
     * @param queryId the queryId to set
     * @since 4.0.0.0
     */
    public Command setQueryId(CommandId queryId)
    {
        QueryId = queryId;
        return this;
    }
    /**
     * Return the command type for this command.
     * @return the command
     * @since 4.0.0.0
     */
    public int getCommand()
    {
        return Command;
    }
    /**
     * Set the command type for this command.
     * @param command the command to set
     * @since 4.0.0.0
     */
    public Command setCommand(int command)
    {
        Command = command;
        return this;
    }
    /**
     * Get the topic for this command.
     * @return the topic
     * @since 4.0.0.0
     */
    public String getTopic()
    {
        return Topic!=null?new String(Topic,_topicOffset,_topicLength):TopicString;
    }
    /**
     * Set the topic for this command.
     * @param topic the topic to set
     * @since 4.0.0.0
     */
    public Command setTopic(String topic)
    {
        TopicString = topic;
        Topic = null;
        return this;
    }
    /**
     * Set the topic for this command from raw bytes.
     * @param topic The raw bytes to be used for the topic.
     * @param offset The offset into topic where the topic data begins
     * @param length The length of the topic.
     * @since 4.0.0.0
     */
    public Command setTopic(byte[] topic, int offset, int length)
    {
        Topic = topic;
        TopicString = null;
        _topicLength = length;
        _topicOffset = 0;
        return this;
    }
    /**
     * Get the filter for this command.
     * @return the filter
     * @since 4.0.0.0
     */
    public String getFilter()
    {
        return Filter;
    }
    /**
     * Set the filter for this command.
     * @param filter the filter to set
     * @since 4.0.0.0
     */
    public Command setFilter(String filter)
    {
        Filter = filter;
        return this;
    }
    /**
     * Get the orderBy clause for this command.
     * @return the orderBy
     * @since 4.0.0.0
     */
    public String getOrderBy()
    {
        return OrderBy;
    }
    /**
     * Set the orderBy clause for this command.
     * @param orderBy the orderBy to set
     * @since 4.0.0.0
     */
    public Command setOrderBy(String orderBy)
    {
        OrderBy = orderBy;
        return this;
    }
    /**
     * Get the subId for this command.
     * @return the subId
     * @since 4.0.0.0
     */
    public CommandId getSubId()
    {
        return SubId;
    }
    /**
     * Set the subId for this command.
     * @param subId the subId to set
     * @since 4.0.0.0
     */
    public Command setSubId(CommandId subId)
    {
        if(subId != null) SubId = subId;
        return this;
    }
    
    /**
     * Set the subId for this command.
     * @param subId the subId to set
     * @since 4.0.0.0
     */
    public Command setSubId(String subId)
    {
        if(subId != null) SubId = new CommandId(subId);
        return this;
    }
    /**
     * Get the bookmark for this command.
     * @return the bookmark
     * @since 4.0.0.0
     */
    public String getBookmark()
    {
        return Bookmark;
    }
    /**
     * Set the bookmark for this command.
     * @param bookmark the bookmark to set
     * @since 4.0.0.0
     */
    public Command setBookmark(String bookmark)
    {
        Bookmark = bookmark;
        return this;
    }
    /**
     * Get the options for this command.
     * @return the options
     * @since 4.0.0.0
     */
    public String getOptions()
    {
        return Options;
    }
    /**
     * Set the options for this command. Options are a comma-delimited list.
     * @param options the options to set
     * @since 4.0.0.0
     */
    public Command setOptions(String options)
    {
        Options = options;
        return this;
    }
    /**
     * Get the ackType for this command.
     * @return the ackType
     * @since 4.0.0.0
     */
    public int getAckType()
    {
        return AckType;
    }
    /**
     * Set the ackType for this command. Different ackTypes are supported
     * for each command. See the Command Reference for details.
     * @param ackType the ackType to set
     * @since 4.0.0.0
     */
    public Command setAckType(int ackType)
    {
        AckType = ackType;
        return this;
    }

    /**
     * Adds an additional ackType to the ackTypes already set
     * for this Command.
     *
     * @param ackType the ackType to add
     * @since 4.0.0.0
     */
    public Command addAckType(int ackType)
    {
        AckType |= ackType;
        return this;
    }
    /**
     * Get the data for this command.
     * @return the data
     * @since 4.0.0.0
     */
    public String getData()
    {
        return Data!=null?new String(Data, _dataOffset, _dataLength):DataString;
    }
    /**
     * Set the data for this command.
     * @param data the data to set
     * @since 4.0.0.0
     */
    public Command setData(String data)
    {
        DataString = data;
        Data = null;
        return this;
    }
    /**
     * Set the data for this command.
     * @param data The raw bytes to be used for the data.
     * @param offset The offset into data where the data begins.
     * @param length The length of the data.
     * @since 4.0.0.0
     */
    public Command setData(byte[] data, int offset, int length)
    {
        DataString = null;
        Data = data;
        _dataOffset = offset;
        _dataLength = length;
        return this;
    }
    /**
     * Get the timeout for this command.
     * @return the timeout
     * @since 4.0.0.0
     */
    public long getTimeout()
    {
        return Timeout;
    }
    /**
     * Set the timeout for this command. Notice that the timeout is
     * monitored in the AMPS client, not on the server.
     * @param timeout the timeout to set
     * @since 4.0.0.0
     */
    public Command setTimeout(long timeout)
    {
        Timeout = timeout;
        return this;
    }
    /**
     * Get the topN value for this command.
     * @return the topN
     * @since 4.0.0.0
     */
    public int getTopN()
    {
        return TopN;
    }
    /**
     * Set the topN value for this command. This parameter sets a maximum
     * number of records returned by a SOW query.
     * @param topN the topN to set
     * @since 4.0.0.0
     */
    public Command setTopN(int topN)
    {
        TopN = topN;
        return this;
    }
    /**
     * Get the batchSize for this command.
     * @return the batchSize
     * @since 4.0.0.0
     */
    public int getBatchSize()
    {
        return BatchSize;
    }
    /**
     * Set the batchSize for this command.
     * @param batchSize the batchSize to set
     * @since 4.0.0.0
     */
    public Command setBatchSize(int batchSize)
    {
        BatchSize = batchSize;
        return this;
    }
    
    /**
     * Get the expiration from this command. This method returns 0 if
     * no expriation is set. Use {@link Command#hasExpiration() } to check if an
     * expiration is set.
     * @return the expiration
     * @since 4.0.0.0
     */
    public int getExpiration()
    {
        if(!_expiration_isSet) return 0;
        return Expiration;
    }
    /**
     * Returns true if this command has an expiration set.
     * @return true if an expiration is set.
     * @since 4.0.0.0
     */
    public boolean hasExpiration()
    {
        return _expiration_isSet;
    }
    /**
     * Set the expiration for this command.
     * @param expiration the expiration to set
     * @since 4.0.0.0
     */
    public Command setExpiration(int expiration)
    {
        Expiration = expiration;
        _expiration_isSet = true;
        return this;
    }
    /**
     * Clears any expiration value set on self.
     * @since 4.0.0.0
     */
    public void unsetExpiration()
    {
        _expiration_isSet = false;
    }
    /**
     * Binds self to a given client, preparing a message from that client to be sent.
     * @param client_ The client to bind against
     * @return The CommandId for this command, if one is set, or null.
     * @since 4.0.0.0
     */
    protected CommandId prepare(Client client_)
    {
        if(_message == null)
        {
            _message = client_.allocateMessage(); 
        }
        else
        {
            _message.reset();
        }
        
        _message.setCommand(Command);
        if(CommandId!=null) _message.setCommandId(CommandId);
        _message.setAckType(AckType);
        if(Bookmark != null) _message.setBookmark(Bookmark);
        if(Data != null)
            _message.setData(Data,_dataOffset,_dataLength);
        else if(DataString != null)
            _message.setData(DataString);
        
        if(Filter!=null) _message.setFilter(Filter);
        if(Options!=null) _message.setOptions(Options);
        if(OrderBy!=null) _message.setOrderBy(OrderBy);
        if(QueryId!=null) _message.setQueryId(QueryId);
        if(SubId!=null) _message.setSubId(SubId);
        if(Topic!=null) 
            _message.setTopic(Topic,_topicOffset,_topicLength);
        else if(TopicString != null)
            _message.setTopic(TopicString);
        if(_expiration_isSet) _message.setExpiration(Expiration);
        if(SOWKeys!=null) _message.setSowKeys(SOWKeys);
        if(isSubscribe()) _message.setSendMatchingIds(true);
        if(BatchSize != 1) _message.setBatchSize(BatchSize);
        if(TopN != -1) _message.setTopN(TopN);
        if(CorrelationId != null) _message.setCorrelationId(CorrelationId);
        return CommandId;
    }
    /**
     * Sets the client sequence number for this command.
     * @since 4.0.0.0
     */
    
    protected Command setClientSequenceNumber(long seqNumber)
    {
        ClientSequenceNumber= seqNumber;
        return this;
    }
    
    /**
     * Get the client sequence number for this command.
     * @return the client sequence number generated for this command, or 0 if none was generated.
     * @since 4.0.0.0
     */
    public long getClientSequenceNumber()
    {
        return ClientSequenceNumber;
    }
    
    /**
     * Sets the correlation Id for this command
     * @since 4.0.0.0
     */
    public Command setCorrelationId(String correlationId_)
    {
        CorrelationId = correlationId_;
        return this;
    }
    /**
     * Returns the correlation Id for this command
     * @since 4.0.0.0
     */
    public String getCorrelationId()
    {
        return CorrelationId;
    }
}
